#!/bin/sh

if [ "$(id -u)" -ne 0 ]; then
	echo "Script $0 can only be run by root." >&2
	exit 1
fi

# Variables used by modules
export cpch_DRY_RUN=0
export cpch_VERBOSITY=0
export cpch_LOGFILE=
export TIMESTAMP=0
# Verbosity levels:
# -2 Suppress erros
# -1 Quiet mode
# 0 Default
# 1 Verbose mode

# Variables only used by driver
NOSHELL=0
LIST=0
STOP_ON_ERR=0

# i.e. the new 'export log log_input err verbose'
export PATH="$PATH:$(realpath "$(dirname "$0")")/ubuntu-modules/bin"

USAGE_STRING="Usage: $0 [-ehlnqstuv] [-L FILE]"
HELP_STRING="${USAGE_STRING}

-e, --stop-on-err    Stop all scripts on first error
-h, --help           Display this help message
    --help-verbosity Display help for verbosity levels
-L FILE, --log=FILE  Copy log output to FILE
-l, --list           List modules to be run only
-n, --no-shell       Never ask to open a shell
-q, --quiet          Lower verbosity level
-s, --dry-run        Dry run
-t, --timestamp      Timestamp log files (requires -L)
-u, --usage          Print Usage
-v, --verbose        Raise verbosity level"

HELP_VERBOSITY="Verbosity levels:
-2 Suppress errors
-1 Quiet mode
 0 Default
 1 Verbose mode"

# Option processing
while getopts 'ehL:lnqstuv' opt; do
	case $opt in
	e) STOP_ON_ERR=1 ;;
	h) echo "$HELP_STRING"; exit 0 ;;
	L) [ ! -e "$OPTARG" ] || [ -w "$OPTARG" ] && cpch_LOGFILE="$OPTARG" || exit 1 ;;
	l) LIST=1 ;;
	n) NOSHELL=1 ;;
	q) cpch_VERBOSITY=$((cpch_VERBOSITY - 1)) ;;
	s) cpch_DRY_RUN=1 ;;
	t) TIMESTAMP=1 ;;
	u) echo "$USAGE_STRING"; exit 0 ;;
	v) cpch_VERBOSITY=$((cpch_VERBOSITY + 1)) ;;
	\?) echo "$USAGE_STRING"; exit 1 ;;
	*) error "getopts issue"; exit 1
	esac
done
shift $((OPTIND - 1))

# Long options
while [ $# -gt 0 ]; do
	case $1 in
	--stop-on-err) STOP_ON_ERR=1 ;;
	--help) echo "$HELP_STRING"; exit 0 ;;
	--log) [ ! -e "$2" ] || [ -w "$2" ] && cpch_LOGFILE="$2" || exit 1; shift ;;
	--log=*) [ ! -e "${1#--log=}" ] || [ -w "${1#--log=}" ] && cpch_LOGFILE="${1#--log=}" || exit 1 ;;
	--list) LIST=1 ;;
	--no-shell) NOSHELL=1 ;;
	--quiet) cpch_VERBOSITY=$((cpch_VERBOSITY - 1)) ;;
	--dry-run) cpch_DRY_RUN=1 ;;
	--timestamp) TIMESTAMP=1 ;;
	--usage) echo "$USAGE_STRING"; exit 0 ;;
	--verbose) cpch_VERBOSITY=$((cpch_VERBOSITY + 1)) ;;
	*) echo "$USAGE_STRING"; exit 1
	esac
	shift
done

case $cpch_VERBOSITY in
-2)
	if [ "$cpch_DRY_RUN" -eq 0 ]; then
		APT_FLAGS=-qq
	else
		APT_FLAGS=-q
	fi ;;
-1) APT_FLAGS=-q ;;
0) APT_FLAGS= ;;
1) APT_FLAGS=-v ;;
esac

if [ "$cpch_DRY_RUN" -eq 1 ]; then
	APT_FLAGS="$APT_FLAGS -s"
fi

export APT_FLAGS="-y $APT_FLAGS"
export APT_UPDATE_FLAGS="${APT_FLAGS%-s}"

# Sanitize blacklist
sed -i '/^$/d' "$(dirname "$0")/ubuntu-modules/blacklist"

for script_name in $(find "$(dirname "$0")/ubuntu-modules" -type f -executable -regex '^.*ubuntu-modules/[0-9][0-9][-a-zA-Z0-9_.]+[^~]$' | sort); do
	if printf $script_name | grep -qf "$(dirname "$0")/ubuntu-modules/blacklist"; then
		continue
	fi

	if [ "$LIST" -eq 1 ]; then
		echo "$script_name"
	else
		if ! "$script_name"; then
			if [ "$STOP_ON_ERR" -eq 1 ]; then
				exit 1
			fi
			if [ "$NOSHELL" -eq 0 ]; then
				echo -n "Shell or continue? [S/c] "
				read answer
				if [ "$answer" = "S" ] || [ "$answer" = "s" ]; then
					"$SHELL"
				fi
			fi
		fi
	fi
done
